/**
  ******************************************************************************
  * @file    py32f410_hal_comp.h
  * @author  MCU Application Team
  * @brief   Header file of COMP HAL module.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __PY32F410_HAL_COMP_H
#define __PY32F410_HAL_COMP_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "py32f410_hal_def.h"
#include "py32f410_ll_exti.h"

/** @addtogroup PY32F410_HAL_Driver
  * @{
  */


/** @addtogroup COMP
  * @{
  */

/* Exported types ------------------------------------------------------------*/
/** @defgroup COMP_Exported_Types COMP Exported Types
  * @{
  */

/**
  * @brief  COMP Init structure definition
  */
typedef struct
{

  uint32_t WindowMode;         /*!< Set window mode of a pair of comparators instances
                                    (2 consecutive instances odd and even COMP<x> and COMP<x+1>).
                                    Note: HAL COMP driver allows to set window mode from any COMP instance of the pair of COMP instances composing window mode.
                                    This parameter can be a value of @ref COMP_WindowMode */

  uint32_t Mode;               /*!< Set comparator operating mode to adjust power and speed.
                                    Note: For the characteristics of comparator power modes
                                          (propagation delay and power consumption), refer to device datasheet.
                                    This parameter can be a value of @ref COMP_PowerMode */

  uint32_t InputPlus;          /*!< Set comparator input plus (non-inverting input).
                                    This parameter can be a value of @ref COMP_InputPlus */

  uint32_t InputMinus;         /*!< Set comparator input minus (inverting input).
                                    This parameter can be a value of @ref COMP_InputMinus */
  
  uint32_t VrefSrc;            /*!< Set comparator Vref Source.
                                    This parameter can be a value of @ref COMP_VrefSrc */
  
  uint32_t VrefDiv;            /*!< Set comparator Vref Div.
                                    This parameter can be a value of @ref COMP_VrefDiv */
  
  uint32_t Hysteresis;         /*!< Set comparator hysteresis mode of the input minus.
                                    This parameter can be a value of @ref COMP_Hysteresis */

  uint32_t OutputPol;          /*!< Set comparator output polarity.
                                    This parameter can be a value of @ref COMP_OutputPolarity */

  uint32_t BlankingSrce;       /*!< Set comparator blanking source.
                                    This parameter can be a value of @ref COMP_BlankingSrce */

  uint32_t TimAndExtiOutSel;   /*!< Set whether the comparator output is filtered to TIM or EXTI. 
                                    This parameter can be a value of @ref COMP_TIM_EXTI_Out_Sel */ 
                                   
  uint32_t DigitalFilter;      /*!< Specifies the digital filter. the filter is prohibited 
                                    when the value is zero.
                                    This parameter must be a number between 0 and 0xFFFF */

  uint32_t TriggerMode;        /*!< Set the comparator output triggering External Interrupt Line (EXTI).
                                    This parameter can be a value of @ref COMP_EXTI_TriggerMode */

} COMP_InitTypeDef;

/**
  * @brief  HAL COMP state machine: HAL COMP states definition
  */
#define COMP_STATE_BITFIELD_LOCK  (0x10U)
typedef enum
{
  HAL_COMP_STATE_RESET             = 0x00U,                                             /*!< COMP not yet initialized                             */
  HAL_COMP_STATE_RESET_LOCKED      = (HAL_COMP_STATE_RESET | COMP_STATE_BITFIELD_LOCK), /*!< COMP not yet initialized and configuration is locked */
  HAL_COMP_STATE_READY             = 0x01U,                                             /*!< COMP initialized and ready for use                   */
  HAL_COMP_STATE_READY_LOCKED      = (HAL_COMP_STATE_READY | COMP_STATE_BITFIELD_LOCK), /*!< COMP initialized but configuration is locked         */
  HAL_COMP_STATE_BUSY              = 0x02U,                                             /*!< COMP is running                                      */
  HAL_COMP_STATE_BUSY_LOCKED       = (HAL_COMP_STATE_BUSY | COMP_STATE_BITFIELD_LOCK)   /*!< COMP is running and configuration is locked          */
} HAL_COMP_StateTypeDef;

/**
  * @brief  COMP Handle Structure definition
  */
#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
typedef struct __COMP_HandleTypeDef
#else
typedef struct
#endif
{
  COMP_TypeDef       *Instance;       /*!< Register base address    */
  COMP_InitTypeDef   Init;            /*!< COMP required parameters */
  HAL_LockTypeDef    Lock;            /*!< Locking object           */
  __IO HAL_COMP_StateTypeDef  State;  /*!< COMP communication state */
  __IO uint32_t      ErrorCode;       /*!< COMP error code */
#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
  void (* TriggerCallback)(struct __COMP_HandleTypeDef *hcomp);   /*!< COMP trigger callback */
  void (* MspInitCallback)(struct __COMP_HandleTypeDef *hcomp);   /*!< COMP Msp Init callback */
  void (* MspDeInitCallback)(struct __COMP_HandleTypeDef *hcomp); /*!< COMP Msp DeInit callback */
#endif /* USE_HAL_COMP_REGISTER_CALLBACKS */
} COMP_HandleTypeDef;

#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
/**
  * @brief  HAL COMP Callback ID enumeration definition
  */
typedef enum
{
  HAL_COMP_TRIGGER_CB_ID                = 0x00U,  /*!< COMP trigger callback ID */
  HAL_COMP_MSPINIT_CB_ID                = 0x01U,  /*!< COMP Msp Init callback ID */
  HAL_COMP_MSPDEINIT_CB_ID              = 0x02U   /*!< COMP Msp DeInit callback ID */
} HAL_COMP_CallbackIDTypeDef;

/**
  * @brief  HAL COMP Callback pointer definition
  */
typedef  void (*pCOMP_CallbackTypeDef)(COMP_HandleTypeDef *hcomp); /*!< pointer to a COMP callback function */

#endif /* USE_HAL_COMP_REGISTER_CALLBACKS */

/**
  * @}
  */

/* Exported constants --------------------------------------------------------*/
/** @defgroup COMP_Exported_Constants COMP Exported Constants
  * @{
  */

/** @defgroup COMP_Error_Code COMP Error Code
  * @{
  */
#define HAL_COMP_ERROR_NONE             (0x00UL)  /*!< No error */
#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
#define HAL_COMP_ERROR_INVALID_CALLBACK (0x01UL)  /*!< Invalid Callback error */
#endif /* USE_HAL_COMP_REGISTER_CALLBACKS */
/**
  * @}
  */

/** @defgroup COMP_WindowMode COMP Window Mode
  * @{
  */
#define COMP_WINDOWMODE_DISABLE                 (0x00000000UL)         /*!< Window mode disable: Comparators instances pair COMP1 and COMP2 are independent */
#define COMP_WINDOWMODE_COMP2_INPUT_PLUS_COMMON (COMP_CSR_WINMODE | COMP_WINDOWMODE_COMP2) /*!< Window mode enable: Comparators instances pair COMP1 and COMP2 have their input plus connected together. The common input is COMP2 input plus (COMP1 input plus is no more accessible). */


/**
  * @}
  */

/** @defgroup COMP_PowerMode COMP power mode
  * @{
  */
/* Note: For the characteristics of comparator power modes                    */
/*       (propagation delay and power consumption),                           */
/*       refer to device datasheet.                                           */
#define COMP_POWERMODE_HIGHSPEED       (0x00000000UL)         /*!< High Speed */
#define COMP_POWERMODE_MEDIUMSPEED     (COMP_CSR_PWRMODE_0)   /*!< Medium Speed */
/**
  * @}
  */

/** @defgroup COMP_InputPlus COMP input plus (non-inverting input)
  * @{
  */ 
#define COMP_INPUT_PLUS_IO0        (0x00000000UL)       /*!< Comparator input plus connected to IO0(pin PA1 for COMP1, pin PA3 for COMP2) */
#define COMP_INPUT_PLUS_IO1        (COMP_CSR_INPSEL)    /*!< Comparator input plus connected to IO1(pin PB1 for COMP1, pin PA9 for COMP2) */
/**
  * @}
  */

/** @defgroup COMP_InputMinus COMP input minus (inverting input)
  * @{
  */
#define COMP_INPUT_MINUS_IO0        (0x00000000UL)                          /*!< Comparator input minus connected to IO0(pin PA0 for COMP1, pin PA2 for COMP2) */
#define COMP_INPUT_MINUS_IO1        (                    COMP_CSR_INNSEL_0) /*!< Comparator input minus connected to IO1(pin PA4 for COMP1, pin PA5 for COMP2) */
#define COMP_INPUT_MINUS_VREFCMP    (COMP_CSR_INNSEL_1                    ) /*!< Comparator input minus connected to IO4(VREFCMP for all) */ 
#define COMP_INPUT_MINUS_TS_VIN     (COMP_CSR_INNSEL_1 | COMP_CSR_INNSEL_0) /*!< Comparator input minus connected to IO5(TS_VIN for all) */ 
/**
  * @}
  */
  
/** @defgroup COMP_VrefSrc COMP Vref Source
  * @{
  */

#define COMP_VREFCMP_SOURCE_VCC              (COMP_CSR_VCSEL) 
#define COMP_VREFCMP_SOURCE_VREFBUF2P048V    (ADC_CCR_VREFBUFEN | 0x00000000U)                                         /* Vrefbuf 2.048V */  
#define COMP_VREFCMP_SOURCE_VREFBUF2P5V      (ADC_CCR_VREFBUFEN | ADC_CCR_VREFBUFSEL_0)                                /* Vrefbuf 2.5V */  
#define COMP_VREFCMP_SOURCE_VREFBUF2P9V      (ADC_CCR_VREFBUFEN | ADC_CCR_VREFBUFSEL_1)                                /* Vrefbuf 2.9V */  
#define COMP_VREFCMP_SOURCE_VREFBUF1P024V    (ADC_CCR_VREFBUFEN | ADC_CCR_VREFBUFSEL_1 | ADC_CCR_VREFBUFSEL_0)         /* Vrefbuf 1.024V */  
#define COMP_VREFCMP_SOURCE_VREFBUF1P5V      (ADC_CCR_VREFBUFEN | ADC_CCR_VREFBUFSEL_2 )                               /* Vrefbuf 1.5V */  

/**
  * @}
  */

/** @defgroup COMP_VrefDiv COMP Vref Div
  * @{
  */
#define COMP_VREFCMP_DIV_DISABLE           (0x00000000)  
#define COMP_VREFCMP_DIV_1_256VREFCMP      (COMP_CSR_VCDIV_EN)  
#define COMP_VREFCMP_DIV_2_256VREFCMP      (COMP_CSR_VCDIV_EN |                                                                                                                                      COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_3_256VREFCMP      (COMP_CSR_VCDIV_EN |                                                                                                                   COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_4_256VREFCMP      (COMP_CSR_VCDIV_EN |                                                                                                                   COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_5_256VREFCMP      (COMP_CSR_VCDIV_EN |                                                                                                COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_6_256VREFCMP      (COMP_CSR_VCDIV_EN |                                                                                                COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_7_256VREFCMP      (COMP_CSR_VCDIV_EN |                                                                                                COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_8_256VREFCMP      (COMP_CSR_VCDIV_EN |                                                                                                COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_9_256VREFCMP      (COMP_CSR_VCDIV_EN |                                                                             COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_10_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                                             COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_11_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                                             COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_12_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                                             COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_13_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                                             COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_14_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                                             COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_15_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                                             COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_16_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                                             COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_17_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4                                                                            )
#define COMP_VREFCMP_DIV_18_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4                                                          | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_19_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_20_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_21_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_22_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_23_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_24_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_25_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_26_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_27_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_28_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_29_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_30_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_31_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_32_256VREFCMP     (COMP_CSR_VCDIV_EN |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_33_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                                                                                               )
#define COMP_VREFCMP_DIV_34_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                                                                             | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_35_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                                                          | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_36_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                                                          | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_37_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_38_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_39_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_40_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_41_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_42_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_43_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_44_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_45_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_46_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_47_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_48_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_49_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                                                            )
#define COMP_VREFCMP_DIV_50_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                                          | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_51_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_52_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_53_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_54_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_55_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_56_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_57_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_58_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_59_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_60_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_61_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_62_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_63_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_64_256VREFCMP     (COMP_CSR_VCDIV_EN |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)

#define COMP_VREFCMP_DIV_65_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6                                                                                                                  )  
#define COMP_VREFCMP_DIV_66_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                                                                                COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_67_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                                                             COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_68_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                                                             COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_69_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                                          COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_70_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                                          COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_71_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                                          COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_72_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                                          COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_73_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_74_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_75_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_76_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_77_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_78_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_79_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_80_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_81_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                                                                            )
#define COMP_VREFCMP_DIV_82_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                                                          | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_83_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_84_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_85_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_86_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_87_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_88_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_89_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_90_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_91_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_92_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_93_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_94_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_95_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_96_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_97_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                                                                               )
#define COMP_VREFCMP_DIV_98_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                                                             | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_99_256VREFCMP     (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                                          | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_100_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                                          | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_101_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_102_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_103_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_104_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_105_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_106_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_107_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_108_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_109_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_110_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_111_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_112_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_113_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                                                            )
#define COMP_VREFCMP_DIV_114_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                                          | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_115_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_116_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_117_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_118_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_119_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_120_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_121_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_122_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_123_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_124_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_125_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_126_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_127_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_128_256VREFCMP    (COMP_CSR_VCDIV_EN |                    COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)

#define COMP_VREFCMP_DIV_129_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7)  
#define COMP_VREFCMP_DIV_130_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                                                                                      COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_131_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                                                                   COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_132_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                                                                   COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_133_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                                                COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_134_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                                                COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_135_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                                                COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_136_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                                                COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_137_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                             COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_138_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                             COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_139_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                             COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_140_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                             COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_141_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                             COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_142_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                             COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_143_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                             COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_144_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                                             COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_145_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4                                                                            )
#define COMP_VREFCMP_DIV_146_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4                                                          | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_147_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_148_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_149_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_150_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_151_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_152_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_153_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_154_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_155_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_156_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_157_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_158_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_159_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_160_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                                          COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_161_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                                                                                               )
#define COMP_VREFCMP_DIV_162_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                                                                             | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_163_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                                                          | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_164_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                                                          | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_165_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_166_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_167_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_168_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_169_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_170_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_171_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_172_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_173_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_174_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_175_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_176_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_177_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                                                            )
#define COMP_VREFCMP_DIV_178_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                                          | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_179_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_180_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_181_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_182_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_183_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_184_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_185_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_186_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_187_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_188_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_189_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_190_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_191_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_192_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 |                                       COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)

#define COMP_VREFCMP_DIV_193_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6                                                                                                                  )  
#define COMP_VREFCMP_DIV_194_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                                                                                COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_195_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                                                             COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_196_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                                                             COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_197_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                                          COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_198_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                                          COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_199_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                                          COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_200_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                                          COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_201_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_202_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_203_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_204_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_205_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_206_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_207_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_208_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                                       COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_209_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                                                                            )
#define COMP_VREFCMP_DIV_210_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                                                          | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_211_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_212_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_213_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_214_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_215_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_216_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_217_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_218_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_219_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_220_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_221_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_222_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_223_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_224_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 |                    COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_225_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                                                                               )
#define COMP_VREFCMP_DIV_226_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                                                             | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_227_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                                          | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_228_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                                          | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_229_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_230_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_231_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_232_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                                       | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_233_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_234_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_235_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_236_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_237_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_238_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_239_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_240_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5                    | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_241_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                                                            )
#define COMP_VREFCMP_DIV_242_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                                          | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_243_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_244_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                                       | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_245_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_246_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_247_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_248_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4                    | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_249_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                                         )
#define COMP_VREFCMP_DIV_250_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                                       | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_251_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_252_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3                    | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_253_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                                      )
#define COMP_VREFCMP_DIV_254_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2                    | COMP_CSR_VCDIV_0)
#define COMP_VREFCMP_DIV_255_256VREFCMP    (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1                   )
#define COMP_VREFCMP_DIV_VREFCMP           (COMP_CSR_VCDIV_EN | COMP_CSR_VCDIV_7 | COMP_CSR_VCDIV_6 | COMP_CSR_VCDIV_5 | COMP_CSR_VCDIV_4 | COMP_CSR_VCDIV_3 | COMP_CSR_VCDIV_2 | COMP_CSR_VCDIV_1 | COMP_CSR_VCDIV_0)
 
/**
  * @}
  */

/** @defgroup COMP_Hysteresis COMP hysteresis
  * @{
  */
#define COMP_HYSTERESIS_DISABLE           (0x00000000UL)                       /*!< No hysteresis */
#define COMP_HYSTERESIS_ENABLE            (0x00000001UL)                       /*!< Hysteresis enable */  
/**
  * @}
  */

/** @defgroup COMP_TIM_EXTI_Out_Sel COMP TIM EXTI Output Selection
  * @{
  */
#define COMP_TIM_EXTI_OUT_NOFILTER                 (0x00000000UL)                      /*!< The comparator output to TIM or EXTI is no filtered */
#define COMP_TIM_EXTI_OUT_FILTER                   (COMP_CSR_TIM_EXTI_OUT_SEL)         /*!< The comparator output to TIM or EXTI is filtered */
/**
  * @} 
  */  

/** @defgroup COMP_OutputPolarity COMP output Polarity
  * @{
  */
#define COMP_OUTPUTPOL_NONINVERTED     (0x00000000UL)         /*!< COMP output level is not inverted (comparator output is high when the input plus is at a higher voltage than the input minus) */
#define COMP_OUTPUTPOL_INVERTED        (COMP_CSR_POLARITY)    /*!< COMP output level is inverted     (comparator output is low  when the input plus is at a higher voltage than the input minus) */
/**
  * @}
  */

/** @defgroup COMP_BlankingSrce  COMP blanking source
  * @{
  */
#define COMP_BLANKINGSRC_NONE       (0x00000000UL)          /*!<Comparator output without blanking */
#define COMP_BLANKINGSRC_TIM1_OC5   (                                            COMP_CSR_BLANKSEL_0)   
#define COMP_BLANKINGSRC_TIM2_OC3   (                      COMP_CSR_BLANKSEL_1                      )   
#define COMP_BLANKINGSRC_TIM3_OC3   (                      COMP_CSR_BLANKSEL_1 | COMP_CSR_BLANKSEL_0)   
#define COMP_BLANKINGSRC_TIM4_OC3   (COMP_CSR_BLANKSEL_2                                            )  
#define COMP_BLANKINGSRC_TIM15_OC1  (COMP_CSR_BLANKSEL_2                       | COMP_CSR_BLANKSEL_0)  
/**
  * @}
  */

/** @defgroup COMP_OutputLevel COMP Output Level
  * @{
  */
/* Note: Comparator output level values are fixed to "0" and "1",             */
/* corresponding COMP register bit is managed by HAL function to match        */
/* with these values (independently of bit position in register).             */

/* When output polarity is not inverted, comparator output is low when
   the input plus is at a lower voltage than the input minus */
#define COMP_OUTPUT_LEVEL_LOW              (0x00000000UL)
/* When output polarity is not inverted, comparator output is high when
   the input plus is at a higher voltage than the input minus */
#define COMP_OUTPUT_LEVEL_HIGH             (0x00000001UL)
/**
  * @}
  */

/** @defgroup COMP_EXTI_TriggerMode COMP output to EXTI
  * @{
  */
#define COMP_TRIGGERMODE_NONE                 (0x00000000UL)                                            /*!< Comparator output triggering no External Interrupt Line */
#define COMP_TRIGGERMODE_IT_RISING            (COMP_EXTI_IT | COMP_EXTI_RISING)                         /*!< Comparator output triggering External Interrupt Line event with interruption, on rising edge */
#define COMP_TRIGGERMODE_IT_FALLING           (COMP_EXTI_IT | COMP_EXTI_FALLING)                        /*!< Comparator output triggering External Interrupt Line event with interruption, on falling edge */
#define COMP_TRIGGERMODE_IT_RISING_FALLING    (COMP_EXTI_IT | COMP_EXTI_RISING | COMP_EXTI_FALLING)     /*!< Comparator output triggering External Interrupt Line event with interruption, on both rising and falling edges */
#define COMP_TRIGGERMODE_EVENT_RISING         (COMP_EXTI_EVENT | COMP_EXTI_RISING)                      /*!< Comparator output triggering External Interrupt Line event only (without interruption), on rising edge */
#define COMP_TRIGGERMODE_EVENT_FALLING        (COMP_EXTI_EVENT | COMP_EXTI_FALLING)                     /*!< Comparator output triggering External Interrupt Line event only (without interruption), on falling edge */
#define COMP_TRIGGERMODE_EVENT_RISING_FALLING (COMP_EXTI_EVENT | COMP_EXTI_RISING | COMP_EXTI_FALLING)  /*!< Comparator output triggering External Interrupt Line event only (without interruption), on both rising and falling edges */
/**
  * @}
  */

/**
  * @}
  */

/* Exported macro ------------------------------------------------------------*/
/** @defgroup COMP_Exported_Macros COMP Exported Macros
  * @{
  */

/** @defgroup COMP_Handle_Management  COMP Handle Management
  * @{
  */

/** @brief  Reset COMP handle state.
  * @param  __HANDLE__  COMP handle
  * @retval None
  */
#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
#define __HAL_COMP_RESET_HANDLE_STATE(__HANDLE__) do{                                                  \
                                                      (__HANDLE__)->State = HAL_COMP_STATE_RESET;      \
                                                      (__HANDLE__)->MspInitCallback = NULL;            \
                                                      (__HANDLE__)->MspDeInitCallback = NULL;          \
                                                    } while(0)
#else
#define __HAL_COMP_RESET_HANDLE_STATE(__HANDLE__) ((__HANDLE__)->State = HAL_COMP_STATE_RESET)
#endif

/**
  * @brief Clear COMP error code (set it to no error code "HAL_COMP_ERROR_NONE").
  * @param __HANDLE__ COMP handle
  * @retval None
  */
#define COMP_CLEAR_ERRORCODE(__HANDLE__) ((__HANDLE__)->ErrorCode = HAL_COMP_ERROR_NONE)

/**
  * @brief  Enable the specified comparator.
  * @param  __HANDLE__  COMP handle
  * @retval None
  */
#define __HAL_COMP_ENABLE(__HANDLE__)              SET_BIT((__HANDLE__)->Instance->CSR, COMP_CSR_EN)

/**
  * @brief  Disable the specified comparator.
  * @param  __HANDLE__  COMP handle
  * @retval None
  */
#define __HAL_COMP_DISABLE(__HANDLE__)             CLEAR_BIT((__HANDLE__)->Instance->CSR, COMP_CSR_EN)
/**
  * @brief  Lock the specified comparator configuration.
  * @note   Using this macro induce HAL COMP handle state machine being no
  *         more in line with COMP instance state.
  *         To keep HAL COMP handle state machine updated, it is recommended
  *         to use function "HAL_COMP_Lock')".
  * @param  __HANDLE__  COMP handle
  * @retval None
  */
#define __HAL_COMP_LOCK(__HANDLE__)                SET_BIT((__HANDLE__)->Instance->CSR, COMP_CSR_LOCK)

/**
  * @brief  Check whether the specified comparator is locked.
  * @param  __HANDLE__  COMP handle
  * @retval Value 0 if COMP instance is not locked, value 1 if COMP instance is locked
  */
#define __HAL_COMP_IS_LOCKED(__HANDLE__)           (READ_BIT((__HANDLE__)->Instance->CSR, COMP_CSR_LOCK) == COMP_CSR_LOCK)

/**
  * @}
  */

/** @defgroup COMP_Exti_Management  COMP external interrupt line management
  * @{
  */

/**
  * @brief  Enable the COMP1 EXTI line rising edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_ENABLE_RISING_EDGE()    LL_EXTI_EnableRisingTrig(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Disable the COMP1 EXTI line rising edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_DISABLE_RISING_EDGE()   LL_EXTI_DisableRisingTrig(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Enable the COMP1 EXTI line falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_ENABLE_FALLING_EDGE()   LL_EXTI_EnableFallingTrig(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Disable the COMP1 EXTI line falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_DISABLE_FALLING_EDGE()  LL_EXTI_DisableFallingTrig(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Enable the COMP1 EXTI line rising & falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_ENABLE_RISING_FALLING_EDGE()   do { \
                                                               LL_EXTI_EnableRisingTrig(COMP_EXTI_LINE_COMP1); \
                                                               LL_EXTI_EnableFallingTrig(COMP_EXTI_LINE_COMP1); \
                                                             } while(0)

/**
  * @brief  Disable the COMP1 EXTI line rising & falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_DISABLE_RISING_FALLING_EDGE()  do { \
                                                               LL_EXTI_DisableRisingTrig(COMP_EXTI_LINE_COMP1); \
                                                               LL_EXTI_DisableFallingTrig(COMP_EXTI_LINE_COMP1); \
                                                             } while(0)

/**
  * @brief  Enable the COMP1 EXTI line in interrupt mode.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_ENABLE_IT()             LL_EXTI_EnableIT(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Disable the COMP1 EXTI line in interrupt mode.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_DISABLE_IT()            LL_EXTI_DisableIT(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Generate a software interrupt on the COMP1 EXTI line.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_GENERATE_SWIT()         LL_EXTI_GenerateSWI(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Enable the COMP1 EXTI line in event mode.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_ENABLE_EVENT()          LL_EXTI_EnableEvent(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Disable the COMP1 EXTI line in event mode.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_DISABLE_EVENT()         LL_EXTI_DisableEvent(COMP_EXTI_LINE_COMP1)


/**
  * @brief  Enable the COMP2 EXTI line rising edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_ENABLE_RISING_EDGE()    LL_EXTI_EnableRisingTrig(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Disable the COMP2 EXTI line rising edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_DISABLE_RISING_EDGE()   LL_EXTI_DisableRisingTrig(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Enable the COMP2 EXTI line falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_ENABLE_FALLING_EDGE()   LL_EXTI_EnableFallingTrig(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Disable the COMP2 EXTI line falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_DISABLE_FALLING_EDGE()  LL_EXTI_DisableFallingTrig(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Enable the COMP2 EXTI line rising & falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_ENABLE_RISING_FALLING_EDGE()   do { \
                                                               LL_EXTI_EnableRisingTrig(COMP_EXTI_LINE_COMP2); \
                                                               LL_EXTI_EnableFallingTrig(COMP_EXTI_LINE_COMP2); \
                                                             } while(0)

/**
  * @brief  Disable the COMP2 EXTI line rising & falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_DISABLE_RISING_FALLING_EDGE()  do { \
                                                               LL_EXTI_DisableRisingTrig(COMP_EXTI_LINE_COMP2); \
                                                               LL_EXTI_DisableFallingTrig(COMP_EXTI_LINE_COMP2); \
                                                             } while(0)

/**
  * @brief  Enable the COMP2 EXTI line in interrupt mode.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_ENABLE_IT()             LL_EXTI_EnableIT(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Disable the COMP2 EXTI line in interrupt mode.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_DISABLE_IT()            LL_EXTI_DisableIT(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Generate a software interrupt on the COMP2 EXTI line.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_GENERATE_SWIT()         LL_EXTI_GenerateSWI(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Enable the COMP2 EXTI line in event mode.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_ENABLE_EVENT()          LL_EXTI_EnableEvent(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Disable the COMP2 EXTI line in event mode.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_DISABLE_EVENT()         LL_EXTI_DisableEvent(COMP_EXTI_LINE_COMP2)


/**
  * @}
  */

/**
  * @}
  */


/* Private types -------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------*/
/** @defgroup COMP_Private_Constants COMP Private Constants
  * @{
  */

/** @defgroup COMP_WindowMode_Instance_Differentiator COMP window mode instance differentiator
  * @{
  */
#define COMP_WINDOWMODE_COMP2          0x00001000U       /*!< COMP window mode using common input of COMP instance: COMP2 */
/**
  * @}
  */

/** @defgroup COMP_ExtiLine COMP EXTI Lines
  * @{
  */
#define COMP_EXTI_LINE_COMP1           (EXTI_IMR_IMR18)  /*!< EXTI line 18 connected to COMP1 output */
#define COMP_EXTI_LINE_COMP2           (EXTI_IMR_IMR19)  /*!< EXTI line 19 connected to COMP2 output */

/**
  * @}
  */

/** @defgroup COMP_ExtiLine COMP EXTI Lines
  * @{
  */
#define COMP_EXTI_IT                        (0x00000001UL)  /*!< EXTI line event with interruption */
#define COMP_EXTI_EVENT                     (0x00000002UL)  /*!< EXTI line event only (without interruption) */
#define COMP_EXTI_RISING                    (0x00000010UL)  /*!< EXTI line event on rising edge */
#define COMP_EXTI_FALLING                   (0x00000020UL)  /*!< EXTI line event on falling edge */
/**
  * @}
  */

/**
  * @}
  */

/* Private macros ------------------------------------------------------------*/
/** @defgroup COMP_Private_Macros COMP Private Macros
  * @{
  */

/** @defgroup COMP_GET_EXTI_LINE COMP private macros to get EXTI line associated with comparators
  * @{
  */
/**
  * @brief  Get the specified EXTI line for a comparator instance.
  * @param  __INSTANCE__  specifies the COMP instance.
  * @retval value of @ref COMP_ExtiLine
  */

#define COMP_GET_EXTI_LINE(__INSTANCE__)    (((__INSTANCE__) == COMP1) ? COMP_EXTI_LINE_COMP1  \
                                              :COMP_EXTI_LINE_COMP2)


/**
  * @}
  */

/** @defgroup COMP_IS_COMP_Definitions COMP private macros to check input parameters
  * @{
  */
#define IS_COMP_WINDOWMODE(__WINDOWMODE__)  (((__WINDOWMODE__) == COMP_WINDOWMODE_DISABLE)                || \
                                             ((__WINDOWMODE__) == COMP_WINDOWMODE_COMP2_INPUT_PLUS_COMMON)  )

#define IS_COMP_POWERMODE(__POWERMODE__)    (((__POWERMODE__) == COMP_POWERMODE_HIGHSPEED)    || \
                                             ((__POWERMODE__) == COMP_POWERMODE_MEDIUMSPEED)    )

#define IS_COMP_INPUT_PLUS(__COMP_INSTANCE__, __INPUT_PLUS__) (((__INPUT_PLUS__) == COMP_INPUT_PLUS_IO0)  || \
                                                               ((__INPUT_PLUS__) == COMP_INPUT_PLUS_IO1))


#define IS_COMP_INPUT_MINUS(__COMP_INSTANCE__, __INPUT_MINUS__) (((__INPUT_MINUS__) == COMP_INPUT_MINUS_IO0)  || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_IO1)  || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_VREFCMP) || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_TS_VIN))

#define IS_COMP_HYSTERESIS(__HYSTERESIS__)  (((__HYSTERESIS__) == COMP_HYSTERESIS_DISABLE)   || \
                                             ((__HYSTERESIS__) == COMP_HYSTERESIS_ENABLE))

#define IS_COMP_VREFSRC(__VREFCMP__)         (((__VREFCMP__) == COMP_VREFCMP_SOURCE_VCC)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_SOURCE_VREFBUF2P048V)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_SOURCE_VREFBUF2P5V)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_SOURCE_VREFBUF2P9V)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_SOURCE_VREFBUF1P024V)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_SOURCE_VREFBUF1P5V))    

#define IS_COMP_VREFDIV(__VREFCMP__)         (((__VREFCMP__) == COMP_VREFCMP_DIV_DISABLE)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_1_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_2_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_3_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_4_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_5_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_6_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_7_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_8_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_9_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_10_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_11_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_12_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_13_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_14_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_15_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_16_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_17_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_18_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_19_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_20_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_21_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_22_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_23_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_24_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_25_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_26_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_27_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_28_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_29_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_30_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_31_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_32_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_33_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_34_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_35_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_36_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_37_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_38_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_39_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_40_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_41_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_42_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_43_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_44_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_45_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_46_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_47_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_48_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_49_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_50_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_51_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_52_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_53_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_54_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_55_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_56_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_57_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_58_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_59_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_60_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_61_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_62_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_63_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_64_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_65_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_66_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_67_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_68_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_69_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_70_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_71_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_72_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_73_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_74_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_75_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_76_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_77_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_78_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_77_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_78_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_79_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_80_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_81_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_82_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_83_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_84_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_85_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_86_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_87_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_88_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_89_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_90_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_91_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_92_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_93_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_94_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_95_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_96_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_97_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_98_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_99_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_100_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_101_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_102_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_103_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_104_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_105_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_106_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_107_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_108_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_109_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_110_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_111_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_112_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_113_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_114_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_115_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_116_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_117_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_118_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_119_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_120_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_121_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_122_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_123_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_124_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_125_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_126_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_127_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_128_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_129_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_130_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_131_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_132_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_133_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_134_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_135_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_136_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_137_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_138_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_139_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_140_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_141_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_142_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_143_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_144_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_145_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_146_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_147_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_148_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_149_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_150_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_151_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_152_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_153_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_154_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_155_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_156_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_157_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_158_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_159_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_160_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_161_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_162_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_163_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_164_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_165_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_166_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_167_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_168_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_169_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_170_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_171_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_172_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_173_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_174_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_175_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_176_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_177_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_178_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_179_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_180_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_181_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_182_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_183_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_184_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_185_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_186_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_187_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_188_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_189_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_190_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_191_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_192_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_193_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_194_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_195_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_196_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_197_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_198_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_199_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_200_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_201_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_202_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_203_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_204_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_205_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_206_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_207_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_208_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_209_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_210_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_211_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_212_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_213_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_214_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_215_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_216_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_217_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_218_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_219_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_220_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_221_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_222_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_223_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_224_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_225_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_226_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_227_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_228_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_229_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_230_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_231_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_232_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_233_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_234_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_235_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_236_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_237_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_238_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_239_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_240_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_241_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_242_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_243_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_244_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_245_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_246_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_247_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_248_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_249_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_250_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_251_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_252_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_253_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_254_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_255_256VREFCMP)   || \
                                             ((__VREFCMP__) == COMP_VREFCMP_DIV_VREFCMP))

#define IS_COMP_OUTPUTPOL(__POL__)          (((__POL__) == COMP_OUTPUTPOL_NONINVERTED) || \
                                             ((__POL__) == COMP_OUTPUTPOL_INVERTED))


#define IS_COMP_BLANKINGSRCE(__BLANKINGSRCE__)          (((__BLANKINGSRCE__) == COMP_BLANKINGSRC_NONE) || \
                                                         ((__BLANKINGSRCE__) == COMP_BLANKINGSRC_TIM1_OC5) || \
                                                         ((__BLANKINGSRCE__) == COMP_BLANKINGSRC_TIM2_OC3) || \
                                                         ((__BLANKINGSRCE__) == COMP_BLANKINGSRC_TIM3_OC3) || \
                                                         ((__BLANKINGSRCE__) == COMP_BLANKINGSRC_TIM4_OC3) || \
                                                         ((__BLANKINGSRCE__) == COMP_BLANKINGSRC_TIM15_OC1))

#define IS_COMP_TIMANDEXTIOUTSEL(__TIMANDEXTIOUTSEL__)          (((__TIMANDEXTIOUTSEL__) == COMP_TIM_EXTI_OUT_NOFILTER) || \
                                                                 ((__TIMANDEXTIOUTSEL__) == COMP_TIM_EXTI_OUT_FILTER))

#define IS_COMP_TRIGGERMODE(__MODE__)       (((__MODE__) == COMP_TRIGGERMODE_NONE)                 || \
                                             ((__MODE__) == COMP_TRIGGERMODE_IT_RISING)            || \
                                             ((__MODE__) == COMP_TRIGGERMODE_IT_FALLING)           || \
                                             ((__MODE__) == COMP_TRIGGERMODE_IT_RISING_FALLING)    || \
                                             ((__MODE__) == COMP_TRIGGERMODE_EVENT_RISING)         || \
                                             ((__MODE__) == COMP_TRIGGERMODE_EVENT_FALLING)        || \
                                             ((__MODE__) == COMP_TRIGGERMODE_EVENT_RISING_FALLING))

#define IS_COMP_OUTPUT_LEVEL(__OUTPUT_LEVEL__) (((__OUTPUT_LEVEL__) == COMP_OUTPUT_LEVEL_LOW)     || \
                                                ((__OUTPUT_LEVEL__) == COMP_OUTPUT_LEVEL_HIGH))

/**
  * @}
  */

/**
  * @}
  */


/* Exported functions --------------------------------------------------------*/
/** @addtogroup COMP_Exported_Functions
  * @{
  */

/** @addtogroup COMP_Exported_Functions_Group1
  * @{
  */

/* Initialization and de-initialization functions  **********************************/
HAL_StatusTypeDef HAL_COMP_Init(COMP_HandleTypeDef *hcomp);
HAL_StatusTypeDef HAL_COMP_DeInit(COMP_HandleTypeDef *hcomp);
void              HAL_COMP_MspInit(COMP_HandleTypeDef *hcomp);
void              HAL_COMP_MspDeInit(COMP_HandleTypeDef *hcomp);

#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
/* Callbacks Register/UnRegister functions  ***********************************/
HAL_StatusTypeDef HAL_COMP_RegisterCallback(COMP_HandleTypeDef *hcomp, HAL_COMP_CallbackIDTypeDef CallbackID,
    pCOMP_CallbackTypeDef pCallback);
HAL_StatusTypeDef HAL_COMP_UnRegisterCallback(COMP_HandleTypeDef *hcomp, HAL_COMP_CallbackIDTypeDef CallbackID);
#endif /* USE_HAL_COMP_REGISTER_CALLBACKS */
/**
  * @}
  */

/* IO operation functions  *****************************************************/
/** @addtogroup COMP_Exported_Functions_Group2
  * @{
  */
HAL_StatusTypeDef HAL_COMP_Start(COMP_HandleTypeDef *hcomp);
HAL_StatusTypeDef HAL_COMP_Stop(COMP_HandleTypeDef *hcomp);
void              HAL_COMP_IRQHandler(COMP_HandleTypeDef *hcomp);
/**
  * @}
  */

/* Peripheral Control functions  ************************************************/
/** @addtogroup COMP_Exported_Functions_Group3
  * @{
  */
HAL_StatusTypeDef HAL_COMP_Lock(COMP_HandleTypeDef *hcomp);
uint32_t          HAL_COMP_GetOutputLevel(COMP_HandleTypeDef *hcomp);
/* Callback in interrupt mode */
void              HAL_COMP_TriggerCallback(COMP_HandleTypeDef *hcomp);
/**
  * @}
  */

/* Peripheral State functions  **************************************************/
/** @addtogroup COMP_Exported_Functions_Group4
  * @{
  */
HAL_COMP_StateTypeDef HAL_COMP_GetState(COMP_HandleTypeDef *hcomp);
uint32_t              HAL_COMP_GetError(COMP_HandleTypeDef *hcomp);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* PY32F410_HAL_COMP_H */

/************************ (C) COPYRIGHT Puya *****END OF FILE****/
